package com.oracle.svm.shadowed.org.bytedeco.javacpp.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import com.oracle.svm.shadowed.org.bytedeco.javacpp.Loader;
import com.oracle.svm.shadowed.org.bytedeco.javacpp.tools.Builder;
import com.oracle.svm.shadowed.org.bytedeco.javacpp.tools.Generator;
import com.oracle.svm.shadowed.org.bytedeco.javacpp.tools.Parser;

/**
 * Makes it possible to define more than one set of properties for each platform.
 * The effective set of properties are taken from all {@link Platform} values in
 * this annotation, but priority is given to values found later in the list, making
 * it possible to define a default set of properties as the first value of the array,
 * and specializing a smaller set of properties for each platform, subsequently.
 * <p>
 * A class with this annotation gets recognized as top-level enclosing class by
 * {@link Loader#getEnclosingClass(Class)}, with the same implications as with
 * the {@link Platform} annotation.
 * <p>
 * Additionally, it is possible to inherit properties from another class also
 * annotated with this annotation, and specialize further for the current class.
 *
 * @see Builder
 * @see Generator
 * @see Loader
 * @see Parser
 *
 * @author Samuel Audet
 */
@Documented @Retention(RetentionPolicy.RUNTIME)
@Target({ElementType.TYPE})
public @interface Properties {
    /** A list of classes from which to inherit properties. */
    Class[] inherit() default {};
    /** A list of platform names to be used as default for {@link #value()}. */
    String[] names() default {};
    /** A list of properties for different platforms. */
    Platform[] value() default {};
    /** The target Java source code file of the {@link Parser}, unless {@link #global()} is set,
        in which case this specifies the target Java package. */
    String target() default "";
    /** The name of a class where to output any global declarations that are not in classes.
        If left empty, considers the {@link #target()} as a class where to put everything. */
    String global() default "";
    /** An optional helper class the {@link Parser} should use as base for the global class.
        Defaults to the class where this annotation was found. */
    String helper() default "";
}
