/*
 * Decompiled with CFR 0.152.
 */
package org.graalvm.visualizer.data.serialization.lazy;

import java.io.IOException;
import java.lang.ref.ReferenceQueue;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.ProviderNotFoundException;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.graalvm.visualizer.data.serialization.lazy.CachedContent;
import org.openide.util.Exceptions;

public final class ZipFileContent
implements SeekableByteChannel,
CachedContent,
AutoCloseable {
    private static final Logger LOG = Logger.getLogger(ZipFileContent.class.getName());
    private static final long LARGE_OFFSET_LIMIT = 0x100000L;
    private final Path archivePath;
    private final NavigableMap<Long, Path> fileParts;
    private final long totalSize;
    private final Map<Path, ExpandedFile> expandedFiles = new WeakHashMap<Path, ExpandedFile>();
    private final FileSystem archiveSystem;
    private long pos;
    private boolean opened;
    private ReadableByteChannel currentPartChannel;
    private final long largeOffsetLimit;

    public ZipFileContent(Path archivePath, ReferenceQueue<?> cleanup) throws IOException {
        this(archivePath, cleanup, Long.getLong(ZipFileContent.class.getName() + ".largeFileSize", 0x100000L));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ZipFileContent(Path archivePath, ReferenceQueue<?> cleanup, long largeLimit) throws IOException {
        LOG.log(Level.FINE, "Creating ZIP content: {0}", archivePath);
        this.archivePath = archivePath;
        URI archiveURI = URI.create("jar:" + archivePath.toUri());
        Class<FileSystems> clazz = FileSystems.class;
        synchronized (FileSystems.class) {
            FileSystem as;
            try {
                as = FileSystems.newFileSystem(archiveURI, Collections.emptyMap());
            }
            catch (FileSystemAlreadyExistsException ex) {
                as = FileSystems.getFileSystem(archiveURI);
            }
            catch (ProviderNotFoundException ex) {
                throw new IOException(ex);
            }
            this.archiveSystem = as;
            ArchiveContentsScanner sc = new ArchiveContentsScanner();
            for (Path root : this.archiveSystem.getRootDirectories()) {
                Files.walkFileTree(root, sc);
            }
            this.largeOffsetLimit = largeLimit;
            ZipFileContent zipFileContent = this;
            synchronized (zipFileContent) {
                this.currentPartChannel = null;
                this.fileParts = sc.fileParts;
                this.totalSize = sc.filePartOffset;
                this.opened = true;
            }
            return;
        }
    }

    @Override
    public String id() {
        return this.archivePath.toString();
    }

    Path getExtractedPart(long offset) {
        Map.Entry<Long, Path> part = this.fileParts.floorEntry(offset);
        if (part == null) {
            return null;
        }
        ExpandedFile f = this.expandedFiles.get(part.getValue());
        return f == null ? null : f.expandedPath;
    }

    FileSystem getFileSystem() {
        return this.archiveSystem;
    }

    private void closePartChannel() throws IOException {
        if (this.currentPartChannel != null) {
            this.currentPartChannel.close();
        }
        this.currentPartChannel = null;
    }

    private boolean openNextPart() throws IOException {
        this.assureOpened();
        Path p = (Path)this.fileParts.get(this.pos);
        if (p == null) {
            return true;
        }
        LOG.log(Level.FINE, "Opening file part {0}:{1}", new Object[]{this.archivePath, p});
        this.currentPartChannel = Files.newByteChannel(p, new OpenOption[0]);
        return false;
    }

    @Override
    public synchronized int read(ByteBuffer dst) throws IOException {
        int read;
        this.assureOpened();
        if (this.pos >= this.totalSize) {
            return -1;
        }
        if (this.currentPartChannel == null && this.openNextPart()) {
            return -1;
        }
        while ((read = this.currentPartChannel.read(dst)) == -1) {
            this.closePartChannel();
            if (!this.openNextPart()) continue;
            return -1;
        }
        this.pos += (long)read;
        return read;
    }

    @Override
    public synchronized boolean isOpen() {
        return this.opened;
    }

    @Override
    public synchronized void close() throws IOException {
        this.closePartChannel();
        this.archiveSystem.close();
        LOG.log(Level.FINE, "Closing zip for {0}", this.archivePath);
        this.deleteExpandedFiles();
        this.fileParts.clear();
        this.opened = false;
    }

    @Override
    public synchronized ReadableByteChannel subChannel(long start, long end) throws IOException {
        this.assureOpened();
        Map.Entry<Long, Path> fe = this.fileParts.floorEntry(start);
        if (fe == null) {
            throw new IOException("Invalid offset");
        }
        long offset = start - fe.getKey();
        long len = end - start;
        ReadableByteChannel ch = this.advanceToPosition(start, fe.getValue(), offset);
        return new LimitedChannel(ch, len);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public synchronized boolean resetCache(long lastReadOffset) {
        if (lastReadOffset >= this.totalSize) {
            try {
                this.close();
            }
            catch (IOException ex) {
                LOG.log(Level.FINE, "Exception during close.", ex);
            }
            return true;
        }
        ZipFileContent zipFileContent = this;
        synchronized (zipFileContent) {
            Iterator<Long> itK = this.fileParts.headMap(lastReadOffset, true).descendingKeySet().iterator();
            while (itK.hasNext()) {
                Long k = itK.next();
                Path zipFile = (Path)this.fileParts.get(k);
                ExpandedFile ef = this.expandedFiles.get(zipFile);
                if (ef == null || ef.cacheMax > lastReadOffset) continue;
                LOG.log(Level.FINE, "Deleting expanded file for {0} : {1}", new Object[]{zipFile, ef.expandedPath});
                ef.close();
                itK.remove();
            }
            return this.fileParts.isEmpty();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ReadableByteChannel advanceToPosition(long start, Path fileInZip, long offset) throws IOException {
        ExpandedFile expF = this.expandedFiles.get(fileInZip);
        if (expF != null && Files.exists(expF.expandedPath, new LinkOption[0])) {
            LOG.log(Level.FINE, "Reading from expanded enry for {0}", fileInZip);
            SeekableByteChannel seek = Files.newByteChannel(expF.expandedPath, StandardOpenOption.READ);
            seek.position(offset);
            return seek;
        }
        if (expF == null) {
            if (offset >= this.largeOffsetLimit) {
                Path tempFile = Files.createTempFile("unpack_", "bgv", new FileAttribute[0]);
                expF = new ExpandedFile(tempFile, start + Files.size(fileInZip), fileInZip);
                Files.copy(fileInZip, tempFile, StandardCopyOption.REPLACE_EXISTING);
                tempFile.toFile().deleteOnExit();
                LOG.log(Level.FINE, "Expanding entry {0} into {1}", new Object[]{fileInZip, tempFile});
                ZipFileContent zipFileContent = this;
                synchronized (zipFileContent) {
                    this.expandedFiles.put(fileInZip, expF);
                }
            } else {
                SeekableByteChannel ch = Files.newByteChannel(fileInZip, StandardOpenOption.READ);
                LOG.log(Level.FINE, "Skipping {1} bytes in entry {0}", new Object[]{fileInZip, offset});
                ByteBuffer buffer = ByteBuffer.allocate(131072);
                while (offset > 0L) {
                    int read;
                    if (offset < (long)buffer.limit()) {
                        buffer.limit((int)offset);
                    }
                    if ((read = ch.read(buffer)) == -1) break;
                    offset -= (long)read;
                    buffer.clear();
                }
                return ch;
            }
        }
        Path realFile = expF.expandedPath;
        SeekableByteChannel sbc = Files.newByteChannel(realFile, StandardOpenOption.READ);
        sbc.position(offset);
        return sbc;
    }

    private void deleteExpandedFiles() {
        LOG.log(Level.FINE, "Deleting all expanded files for {0}", this.archivePath);
        for (ExpandedFile ef : this.expandedFiles.values()) {
            ef.close();
        }
        this.expandedFiles.clear();
    }

    private synchronized void assureOpened() throws IOException {
        if (!this.opened) {
            throw new ClosedChannelException();
        }
    }

    @Override
    public synchronized long position() throws IOException {
        return this.pos;
    }

    @Override
    public long size() throws IOException {
        return this.totalSize;
    }

    @Override
    public int write(ByteBuffer src) throws IOException {
        throw new UnsupportedOperationException("Unsupported.");
    }

    @Override
    public SeekableByteChannel position(long newPosition) throws IOException {
        throw new UnsupportedOperationException("Unsupported.");
    }

    @Override
    public SeekableByteChannel truncate(long size) throws IOException {
        throw new UnsupportedOperationException("Unsupported.");
    }

    static class ArchiveContentsScanner
    extends SimpleFileVisitor<Path> {
        final Map<Path, Long> offsets = new HashMap<Path, Long>();
        final NavigableMap<Long, Path> fileParts = new TreeMap<Long, Path>();
        private long filePartOffset;

        ArchiveContentsScanner() {
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Path pn = file.getFileName();
            if (pn == null) {
                return FileVisitResult.CONTINUE;
            }
            String fn = pn.toString().toLowerCase(Locale.ENGLISH);
            if (!fn.endsWith(".bgv")) {
                return FileVisitResult.CONTINUE;
            }
            long fs = Files.size(file);
            this.offsets.put(file, this.filePartOffset);
            this.fileParts.put(this.filePartOffset, file);
            LOG.log(Level.FINER, "Found file: {0}, start {1}, size {2}", new Object[]{file, this.filePartOffset, fs});
            this.filePartOffset += fs;
            return FileVisitResult.CONTINUE;
        }
    }

    static final class ExpandedFile {
        private final Path expandedPath;
        private final Path zipPath;
        private final long cacheMax;

        public ExpandedFile(Path expandedPath, long cacheMax, Path zp) {
            this.expandedPath = expandedPath;
            this.cacheMax = cacheMax;
            this.zipPath = zp;
        }

        private void close() {
            try {
                Files.deleteIfExists(this.expandedPath);
            }
            catch (IOException ex) {
                LOG.log(Level.FINE, "Could not delete temp file", Exceptions.attachSeverity((Throwable)ex, (Level)Level.INFO));
            }
        }

        public String toString() {
            return this.zipPath + " => " + this.expandedPath + "[:" + this.cacheMax + "]";
        }
    }

    static final class LimitedChannel
    implements SeekableByteChannel {
        private final ReadableByteChannel delegate;
        private final long limit;
        private long count;

        public LimitedChannel(ReadableByteChannel delegate, long limit) {
            this.delegate = delegate;
            this.limit = limit;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public int read(ByteBuffer dst) throws IOException {
            if (this.count >= this.limit) {
                return -1;
            }
            long rem = this.limit - this.count;
            if ((long)dst.remaining() <= rem) {
                int x = this.delegate.read(dst);
                this.count += (long)x;
                return x;
            }
            int l = dst.limit();
            dst.limit(dst.position() + (int)rem);
            try {
                int x = this.delegate.read(dst);
                if (x > 0) {
                    this.count += (long)x;
                }
                int n = x;
                return n;
            }
            finally {
                dst.limit(l);
            }
        }

        @Override
        public boolean isOpen() {
            return this.delegate.isOpen();
        }

        @Override
        public void close() throws IOException {
            this.delegate.close();
        }

        @Override
        public long position() throws IOException {
            return this.count;
        }

        @Override
        public SeekableByteChannel position(long newPosition) throws IOException {
            if (newPosition > this.limit) {
                throw new IOException();
            }
            if (this.delegate instanceof SeekableByteChannel) {
                return ((SeekableByteChannel)this.delegate).position(newPosition);
            }
            throw new IOException();
        }

        @Override
        public long size() throws IOException {
            return this.limit;
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            throw new UnsupportedOperationException("Not supported.");
        }

        @Override
        public SeekableByteChannel truncate(long size) throws IOException {
            throw new UnsupportedOperationException("Not supported");
        }
    }
}

